/*
*  Arnold emulator (c) Copyright, Kevin Thacker 1995-2015
*
*  This file is part of the Arnold emulator source code distribution.
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#ifndef __LABELSET_HEADER_INCLUDED__
#define __LABELSET_HEADER_INCLUDED__

#include <string>
#include <vector>
#include "../cpc.h"

/* TODO: An address description needs to include some way to identify the memory region */
/* TODO: instead of returning pointers, could return ids, this means we don't handle pointers outside this class directly
and less likely to have problems and crash */
class Label
{
public:
	Label()
	{
		m_bActive = true;
	}
	const std::string getName() const { return m_sName; }
	void setName(const std::string &name) { m_sName = name; }

	const bool isActive() const { return m_bActive; }
	void setActive(bool bState) { m_bActive = bState; }

	Z80_WORD getAddress() const { return m_Address; }
	void setAddress(Z80_WORD Address) { m_Address = Address; }
private:
	/* name of label */
	std::string m_sName;
	/* true if label can be queried */
	 bool m_bActive;
	/* address of label */
	Z80_WORD m_Address;


	Label(const Label &);
	Label &operator=(const Label &);
	bool operator==(const Label &);
};


class LabelSet
{
public:
	typedef std::vector<Label *> LabelsType;
	LabelSet()
	{
		m_bActive = true;
	}
	~LabelSet()
	{
		LabelsType::iterator iter = m_labels.begin();
		for (; iter != m_labels.end(); ++iter)
		{
			delete (*iter);
		}
		m_labels.clear();
	}

	bool getLabelNameByAddr(const Z80_WORD Address, std::string &sName) const;
	bool getLabelAddrByName(const std::string &sName, Z80_WORD &Addr) const;
	const std::string getName() const { return m_sName; }
	void setName(const std::string &name) { m_sName = name; }
	const std::string getFilename() const { return m_sFilename; }
	void setFilename(const std::string &sFilename) { m_sFilename = sFilename; }

	const bool isActive() const { return m_bActive; }
	void setActive(bool bState) { m_bActive = bState; }

	void deleteLabel(Label *label);
	LabelsType getLabels() { return m_labels; }
	Label *addLabel(const std::string &sName, Z80_WORD Addr);
private:
	bool m_bActive; /* if active, then labelset can be quiered */
	std::string m_sName; /* user readable name of labelset */
	std::string m_sFilename; /* path to filename */
	LabelsType m_labels;
	
	LabelSet(const LabelSet &);
	LabelSet &operator=(const LabelSet &);
	bool operator==(const LabelSet &);
};


class LabelSetManager
{
public:
	typedef std::vector<LabelSet *> LabelSetsType;
	LabelSetManager()
	{
		m_current = nullptr;
		m_selected = nullptr;
	}
	~LabelSetManager()
	{
		LabelSetsType::iterator iter = m_labelsets.begin();
		for (; iter != m_labelsets.end(); ++iter)
		{
			delete (*iter);
		}
		m_labelsets.clear();
		m_current = nullptr;
	}
	void deleteLabelSet(LabelSet *labelSetToDelete);
	LabelSetsType getLabelSets() { return m_labelsets; }
	bool getLabelNameByAddr(const Z80_WORD Address, std::string &sName) const;
	bool getLabelAddrByName(const std::string &sName, Z80_WORD &Addr) const;
	//const Label *findLabelByName(const std::stringsName);
	//const Label *findLabelByAddr(Z80_WORD Address);
	bool addLabelToCurrentLabelSet(const std::string &sName, Z80_WORD Address);
	void setCurrentLabelSet(LabelSet *labelset) { m_current = labelset; }
	LabelSet *addEmptyLabelSet(const std::string &sName);
	LabelSet *findLabelSetByName(const std::string &sName);
	void setSelectedLabelSet(LabelSet *labelset) { m_selected = labelset; }
private:
	LabelSet *m_current;
	LabelSet *m_selected;
	LabelSetsType m_labelsets;

	LabelSetManager(const LabelSetManager &);
	LabelSetManager &operator=(const LabelSetManager &);
	bool operator==(const LabelSetManager &);
};

const LabelSetManager &getGlobalLabelSetManager();
LabelSetManager &getGlobalLabelSetManagerRef();

#endif
